/** @file
 *
 * ADIT Pixmap API implementation.
 *
 * @copyright	(c) 2013 ADIT Corporation
 * @author	Vincent Stehle <vincent.stehle@freescale.com
 *
 * TODO: "Cache" surface informations at create_buffer time, rather than
 *       retrieving with accessors?
 */

/**
 * @defgroup buffer_handling Buffer Handling
 * Functions that work on APX buffers
 */

/**
 * @defgroup setup Library setup and shutdown
 * Functions necessary to setup the APX library
 */

#include "apx.h"
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <HAL/gc_hal_base.h>
#include <HAL/gc_hal_raster.h>
#include <wayland-client.h>
#include <wayland-viv-client-protocol.h>
#include "ilm-wl-drm-client-protocol.h"
#include <time.h>
#include <errno.h>

/* Internal structure for APX */
typedef struct _ApxPrivate
{
    struct wl_viv *viv;
    struct ilm_wl_drm *ilm_wl_drm;
    struct wl_display* display;
    struct wl_display *display_wrapper;
    struct wl_compositor *compositor;
    struct wl_surface *surface;
    struct wl_surface *surface_wrapper;
    struct wl_event_queue* queue;
    int redraw_done;
    struct wl_callback* frame_callback;
    int authenticated;
    unsigned int surface_id;

    unsigned refcount;
} ApxPrivate;

#ifdef DEBUG
# define error(FORMAT, ...)	fprintf(stderr, "[Error]: %s: " FORMAT "\n", __func__, ## __VA_ARGS__)
# define info(FORMAT, ...)	fprintf(stderr, "[Info] : %s: " FORMAT "\n", __func__, ## __VA_ARGS__)
# define debug(FORMAT, ...)	fprintf(stderr, "[Debug]: %s: " FORMAT "\n", __func__, ## __VA_ARGS__)
#else
# define error(FORMAT, ...)	gcoOS_DebugTrace(gcvLEVEL_ERROR, "%s: " FORMAT "\n", __func__, ## __VA_ARGS__)
# define info(FORMAT, ...)	gcoOS_DebugTrace(gcvLEVEL_INFO,   "%s: " FORMAT "\n", __func__, ## __VA_ARGS__)
/*# define debug(FORMAT, ...)	gcoOS_DebugTrace(gcvLEVEL_VERBOSE, "%s: " FORMAT "\n", __func__, ## __VA_ARGS__) */
# define debug(FORMAT, ...) do { } while (0)
#endif

static void
set_opaque_region(struct wl_surface *wl_surface, int width, int height, struct wl_compositor* compositor);

static ApxPrivate *apx_private_new ()
{
    return calloc(sizeof(ApxPrivate), 1);
}

static ApxPrivate *apx_private_ref(ApxPrivate *p)
{
    if (p)
    {
        __sync_fetch_and_add(&(p->refcount), 1);
    }
    else
    {
        error("Invalid pointer");
    }

    return p;
}

static int apx_private_unref(ApxPrivate *p)
{
    if (!p)
    {
        error("Invalid global private!");
        return -1;
    }

    if (__sync_sub_and_fetch (&(p->refcount), 1) > 0)
        return 1;

    return 0;
}


#define APX_UNUSED(x) ((void)(x))

struct apx_buffer {
	gcoSURF surf[2];			/* Handle to the GC Surface. */
	struct wl_buffer *wl_buf[2];	/* Wayland buffer. */
	gctPOINTER memory[3];		/* When mapped, pointer(s) to memory. */
    int current;                /* Buffer currently ready for rendering into */
	int locked[2];			/* When true, means the Wayland buffer has been released. */
    gctBOOL double_buffered; /* Whether the buffer has a back-buffer */
    gctBOOL no_copy; /* Whether the buffer should perform a buffer copy on apx_buffer_commit() */
    struct {
        ApxUnlockCallback func; /* Call-back called if compositor releases buffer */
        ApxCallbackDestroyNotify notify; /* Call-back called on data if buffer is destroyed */
        void *data; /* User data for func */
    } callback; /* Unlock callback information */
    struct apx *a; /* Apx context */
};

#ifdef DEBUG
static void diff_ts(struct timespec start, struct timespec end, struct timespec *temp)
{
    if ((end.tv_nsec-start.tv_nsec)<0) {
        temp->tv_sec = end.tv_sec-start.tv_sec-1;
        temp->tv_nsec = 1000000000+end.tv_nsec-start.tv_nsec;
    } else {
        temp->tv_sec = end.tv_sec-start.tv_sec;
        temp->tv_nsec = end.tv_nsec-start.tv_nsec;
    }
}
#endif

static int swap_and_blit(ApxPrivate *priv, struct apx_buffer *b)
{
    APX_UNUSED(priv);
#ifdef DEBUG
    struct timespec before;
    struct timespec after;
    struct timespec diff;
#endif
    int next = (b->current + 1) % 2;

    if (!b->no_copy)
    {
        debug ("Blitting from %d to %d", b->current, next);
        gctUINT32 srcPhy, dstPhy;
        gctPOINTER srcMem, dstMem;
        gctUINT srcWidth, srcHeight;
        gctINT srcStride;
        gctUINT dstWidth, dstHeight;
        gctINT dstStride;
        gceSURF_FORMAT format;

#ifdef DEBUG
        clock_gettime(CLOCK_MONOTONIC, &before);
#endif

        gcoSURF_GetAlignedSize(b->surf[next], &dstWidth, &dstHeight, &dstStride);
        gcoSURF_GetFormat(b->surf[next], gcvNULL, &format);
        gcoSURF_GetAlignedSize(b->surf[b->current], &srcWidth, &srcHeight, &srcStride);

        if (gcoSURF_Lock(b->surf[next], &dstPhy, &dstMem) == gcvSTATUS_OK)
        {
            if (gcoSURF_Lock(b->surf[b->current], &srcPhy, &srcMem) == gcvSTATUS_OK)
            {
                /* Do Nothing */
            }
            else
            {
                error("Failed to lock source surface");
                gcoSURF_Unlock(b->surf[next], &dstMem);
            }
        }
        else
        {
            error("Failed to lock destination surface");
        }

        if (srcMem != gcvNULL && dstMem != gcvNULL)
        {
            gctBOOL doMemcpy = gcvFALSE;
            gcsRECT dstRect = { 0, 0, dstWidth, dstHeight };
            gco2D engine = gcvNULL;
            int status = gcvSTATUS_OK;

            status = gcoHAL_Get2DEngine(gcvNULL, &engine);
            if (status != gcvSTATUS_OK)
            {
                error("Failed to get engine: %d", status);
                doMemcpy = gcvTRUE;
            }

            status = gco2D_SetClipping(engine, &dstRect);
            if (status != gcvSTATUS_OK)
            {
                error("Failed to set clipping: %d. Cannot copy.", status);
                doMemcpy = gcvTRUE;
            }

            status = gco2D_SetColorSource(engine, srcPhy, srcStride, format,
                                          gcvSURF_0_DEGREE, srcWidth, gcvFALSE,
                                          gcvSURF_OPAQUE, 0);
            if (status != gcvSTATUS_OK)
            {
                error("Failed to set color source: %d. Cannot copy.", status);
                doMemcpy = gcvTRUE;
            }

            status = gco2D_SetSource(engine, &dstRect);
            if (status != gcvSTATUS_OK)
            {
                error("Failed to set source: %d. Cannot copy.", status);
                doMemcpy = gcvTRUE;
            }

            status = gco2D_SetTarget(engine, dstPhy, dstStride, gcvSURF_0_DEGREE,
                                     dstWidth);
            if (status != gcvSTATUS_OK)
            {
                error("Failed to set target: %d. Cannot copy.", status);
                doMemcpy = gcvTRUE;
            }

            status = gco2D_Blit(engine, 1, &dstRect, 0xCC, 0xAA, format);
            if (status != gcvSTATUS_OK)
            {
                error("Failed to blit: %d. Cannot copy.", status);
                doMemcpy = gcvTRUE;
            }

            status = gco2D_Flush(engine);
            if (status != gcvSTATUS_OK)
            {
                error("Failed to flush 2D pipeline: %d. Cannot copy.", status);
                doMemcpy = gcvTRUE;
            }

            status = gcoHAL_Commit(gcvNULL, gcvTRUE);
            if (status != gcvSTATUS_OK)
            {
                error("Failed commit hal. Cannot copy.");
                doMemcpy = gcvTRUE;
            }

            if (doMemcpy == gcvTRUE)
            {
                debug("Something went wrong, going to memcpy.");
                memcpy(dstMem, srcMem, srcHeight * srcStride);
            }

            gcoSURF_Unlock(b->surf[b->current], srcMem);
            gcoSURF_Unlock(b->surf[next], dstMem);

#ifdef DEBUG
            clock_gettime(CLOCK_MONOTONIC, &after);
            diff_ts(before, after, &diff);
            debug("CopyPixels;%ld,%09ld;%ld,%09ld;%ld,%09ld",
                  before.tv_sec, before.tv_nsec,
                  after.tv_sec, after.tv_nsec,
                  diff.tv_sec, diff.tv_nsec);
#endif
        }
    }

	b->current = next;

	return 0;
}

static int set_wl_context(struct apx *a, struct wl_display* d)
{
    ApxPrivate *p = NULL;

    p = (ApxPrivate *)a->viv;
    p->display = d;

    p->queue = wl_display_create_queue(p->display);
    if (p->queue == NULL)
    {
        error("wl_display_create_queue failed!");
        return -1;
    }

    p->display_wrapper = wl_proxy_create_wrapper(p->display);
    if (p->display_wrapper == NULL)
    {
        error("wl_proxy_create_wrapper failed!");
        return -1;
    }

    wl_proxy_set_queue((struct wl_proxy *) p->display_wrapper, p->queue);
    wl_proxy_set_queue((struct wl_proxy *) p->viv, p->queue);

    if(p->ilm_wl_drm)
        wl_proxy_set_queue((struct wl_proxy *) p->ilm_wl_drm, p->queue);

    /* There is no frame that we are waiting for */
    p->redraw_done = 1;
    p->frame_callback = NULL;

    return 0;
}

static int set_wl_surface(struct apx *a, struct wl_surface* s, int width, int height)
{
    ApxPrivate *p = NULL;

    p = (ApxPrivate *)a->viv;
    p->surface = s;
    p->surface_id = wl_proxy_get_id((struct wl_proxy*)p->surface);

    p->surface_wrapper = wl_proxy_create_wrapper(p->surface);
    if (p->surface_wrapper == NULL)
    {
        error("wl_proxy_create_wrapper failed!");
        return -1;
    }

    wl_proxy_set_queue((struct wl_proxy *)p->surface_wrapper, p->queue);

    set_opaque_region(p->surface, width, height, p->compositor);

    return 0;
}

static void client_authenticated (void *data, struct ilm_wl_drm *ilm_wl_drm)
{
	ApxPrivate *p = NULL;

	APX_UNUSED(ilm_wl_drm);
	p = (ApxPrivate*)data;
	p->authenticated = 0x1;
}

static const struct ilm_wl_drm_listener ilm_wl_drm_listener ={
	client_authenticated,
};

/**
 * ADIT Pixmap Wayland global handler.
 *
 * Call this in Wayland display registry listener global handler, where all
 * `strcmp(interface)` and `wl_registry_bind()` calls are done.
 *
 * This is where we bind to the `wl_viv` interface.
 *
 * This needs to be called first, before `apx_init()`, as `apx_init()` will
 * check that we are bound to `wl_viv`.
 * @ingroup setup
 *
 * @note Please make sure the passed struct apx is cleared properly using memset
 * or similar methods.
 *
 * @param[in]	data		User data pointer; see `wl_registry_listener`.
 *				Caller must pass a `struct apx` context pointer here.
 *
 * @param[in]	wl_registry	Global registry object; see
 *				`wl_registry_listener`.
 *
 * @param[in]	name		Registry object "name"; see
 *				`wl_registry_listener`.
 *
 * @param[in]	interface	Registry object interface name; see
 *				`wl_registry_listener`.
 *
 * @param[in]	version		Registry object interface version; see
 *				`wl_registry_listener`.
 *
 * @retval	>=0		if success.
 * @retval	<0		if error.
 */
int apx_global_handler(void *data, struct wl_registry *wl_registry, uint32_t name,
			const char *interface, uint32_t version)
{
	struct apx *a = (struct apx *)data;
    ApxPrivate *p = NULL;

	if (!a) {
		error("Bad apx pointer");
		return -1;
	}

	if(!a->viv)
	{
		p = apx_private_new();
		if (!p) {
			error("calloc error");
			return -1;
		}
		a->viv = p;
	}
	else
	{
		p = a->viv;
	}

	if (!strcmp("wl_viv", interface)) {
		extern const struct wl_interface wl_viv_interface;

		if (version != 1) {
			error("bad wl_viv version %u, expected 1", version);
			return -1;
		}

		p->viv = (struct wl_viv *)wl_registry_bind(wl_registry, name,
				&wl_viv_interface, version);

		if (!p->viv) {
			error("wl_registry_bind(wl_viv) error");
			return -1;
		}
	}

	if (!strcmp("ilm_wl_drm", interface)) {
		p->ilm_wl_drm = (struct ilm_wl_drm *)wl_registry_bind(wl_registry, name,
					&ilm_wl_drm_interface, version);

		if (!p->ilm_wl_drm) {
			error("wl_registry_bind(ilm_wl_drm) error");
			return -1;
		}

		if(ilm_wl_drm_add_listener(p->ilm_wl_drm, &ilm_wl_drm_listener, p))
		{
			error("ilm_wl_drm_add_listener failed");
			return -1;
		}
	}

	if (!strcmp("wl_compositor", interface)) {
		p->compositor = (struct wl_compositor *)wl_registry_bind(wl_registry, name,
					&wl_compositor_interface, version);

		if (!p->compositor) {
			error("wl_registry_bind(wl_compositor) error");
			return -1;
		}
	}

	return 0;
}

/**
 * ADIT Pixmap API to do-drm authentication for client.
 *
 * Call this last, if the application is non-root. For the root user
 * this API is not needed to be called
 *
 * @ingroup setup
 * @param[in]	 apxCtx	API context.
 *
 * @param[in]	 display	Pointer to Wayland display.
 *
 * @retval	>=0	if success.
 * @retval	<0	if error.
 */
int apx_authenticate(struct apx *a, struct wl_display *d, unsigned int drm_magic)
{
	ApxPrivate *p = NULL;

	if (!a)
	{
		error("Bad APX pointer!");
		return -1;
	}

	if (!a->viv) {
		error("No priv data");
		return -1;
	}

	p = (ApxPrivate *)a->viv;
	if(!p->ilm_wl_drm)
	{
		error("WARN:wl_drm is not available, so client can't be authenticated");
		return 0;
	}

	/* Set wl_context if necessary. */
	if (!p->display && set_wl_context(a, d) < 0) {
		error("set_wl_context error");
		return -1;
	}

	ilm_wl_drm_authenticate(p->ilm_wl_drm, drm_magic);
	wl_display_roundtrip_queue(p->display, p->queue);
	if(!p->authenticated)
	{
		struct wl_interface *interface;
		uint32_t id;
		uint32_t errcode;
		int err;

		err = wl_display_get_error(p->display);
		if(err == EPROTO)
		{
			errcode = wl_display_get_protocol_error(p->display, (const struct wl_interface**)&interface, &id);
			if(ILM_WL_DRM_ERROR_AUTHENTICATE_FAIL == errcode)
				error("ilm_wl_drm_authenticate errcode:ILM_WL_DRM_ERROR_AUTHENTICATE_FAIL");
		}
		else
		{
			error("ilm_wl_drm_authenticate failed but no valid error from ptotocol");
		}

		return -1;
	}

	return 0;
}

/**
 * De-initialize ADIT Pixmap API.
 *
 * Call this last, once.
 *
 * @ingroup setup
 * @param[in]	a	API context. Currently unused but passed for consistency.
 *
 * @retval	>=0	if success.
 * @retval	<0	if error.
 */
int apx_deinit(struct apx *a)
{
    ApxPrivate *p = NULL;
    /* Note: let the viv pointer as it is, to enable subsequent calls to
       apx_init(). */

    if (!a)
    {
        error("Bad APX pointer!");
        return -1;
    }

    p = (ApxPrivate *)a->viv;

    apx_private_unref(p);
    if (!p->refcount) {
        //Refcount is equal to 0.
        if (p->display_wrapper)
            wl_proxy_wrapper_destroy(p->display_wrapper);

        if (p->surface_wrapper)
            wl_proxy_wrapper_destroy(p->surface_wrapper);

        if (p->viv)
            wl_viv_destroy(p->viv);

        if (p->queue)
            wl_event_queue_destroy(p->queue);

        free(p);
    }

    return 0;
}

/**
 * Initialize ADIT Pixmap API.
 *
 * Call this second first, just after all calls to `apx_global_handler()`,
 * once.
 *
 * This needs to be called after `apx_global_handler()`, as it checks that we
 * are indeed bound to `wl_viv`.
 *
 * @ingroup setup
 *
 * @param[in]	a	API context.
 *
 * @retval	>=0	if success.
 * @retval	<0	if error.
 */
int apx_init(struct apx *a)
{
    ApxPrivate *p = NULL;

	if (!a) {
		error("Bad apx pointer");
		return -1;
	}

	if (!a->viv) {
		error("No wl_viv");
		return -1;
	}

    p = (ApxPrivate *)a->viv;

    apx_private_ref(p);

	return 0;
}

/**
 * Map a buffer for user process use.
 *
 * @param[in]	b	Buffer.
 *
 * @param[out]	p	Pointer to one to three plane pointer(s) virtual
 *			addresses. For RGB based formats, only the first
 *			pointer is relevant.
 *
 * @retval	>=0	if success.
 * @retval	<0	if error.
 * @ingroup buffer_handling
 */
int apx_buffer_map(struct apx_buffer *b, void **p)
{
	if (!b) {
		error("Bad pointer");
		return -1;
	}

	if (gcoSURF_Lock(b->surf[b->current], NULL/*Address*/, b->memory) != gcvSTATUS_OK) {
		error("gcoSURF_Lock error");
		return -1;
	}

	p[0] = b->memory[0];
	p[1] = b->memory[1];	/* TODO: assign 1 & 2 only for YUV formats.*/
	p[2] = b->memory[2];
	return 0;
}

/**
 * Unmap a previously mapped buffer.
 *
 * @param[in]	b	Buffer.
 *
 * @retval	>=0	if success.
 * @retval	<0	if error.
 * @ingroup buffer_handling
 */
int apx_buffer_unmap(struct apx_buffer *b)
{
	if (!b) {
		error("Bad pointer");
		return -1;
	}

	if (!b->memory[0]) {
		error("Bad buffer not previously locked");
		return -1;
	}

	if (gcoSURF_Unlock(b->surf[b->current], b->memory[0]) != gcvSTATUS_OK) {
		error("gcoSURF_Unlock error");
		return -1;
	}

	b->memory[0] = b->memory[1] = b->memory[2] = NULL;
	return 0;
}

static int wayland_sync(struct apx *a)
{
	ApxPrivate *p = NULL;
        int ret = 0;

	p = (ApxPrivate *)a->viv;

	if (!p->display)
		return ret;

	while (!p->redraw_done && ret >= 0) {
                ret = wl_display_dispatch_queue(p->display, p->queue);
	}

	if (p->frame_callback && ret == -1 && !p->redraw_done) {
		wl_callback_destroy(p->frame_callback);
	}

	return ret;
}

/**
 * Destroy buffer.
 *
 * Note that we may be called with incomplete buffers on error paths, or with
 * not-yet-wrapped buffers, too.
 *
 * When the given Wayland display pointer is NULL, this is the caller's
 * responsibility to make sure the buffer is not in use by the compositor.
 *
 * @param[in]	b	Buffer.
 *
 * @param[in]	d	Pointer to Wayland display. Ignored if NULL. If
 *			non-NULL, we wait until we receive the buffer `release`
 *			event.
 *
 * @retval	>=0	if success.
 * @retval	<0	if error.
 * @ingroup buffer_handling
 */
int apx_buffer_destroy(struct apx_buffer *b, struct wl_display *d)
{
	struct apx *a = NULL;
	ApxPrivate *p = NULL;
	int r = 0;

	APX_UNUSED(d);

	r = wayland_sync(b->a);

	if (!b) {
		error("Bad pointer");
		return -1;
	}

	if (b->memory[0] && apx_buffer_unmap(b) < 0) {
		error("apx_buffer_unmap error");
		r = -1;
	}

	if (b->wl_buf[0]) {
		wl_buffer_destroy(b->wl_buf[0]);
	}

	if (b->wl_buf[1]) {
		wl_buffer_destroy(b->wl_buf[1]);
	}

	if (b->surf[0] && gcoSURF_Destroy(b->surf[0]) != gcvSTATUS_OK) {
		error("gcoSURF_Destroy error");
		r = -1;
	}

	if (b->surf[1] && gcoSURF_Destroy(b->surf[1]) != gcvSTATUS_OK) {
		error("gcoSURF_Destroy error");
		r = -1;
	}
	/* Schedule for free of gco surface */
	gcoHAL_Commit(gcvNULL, gcvTRUE);

	a = b->a;
	p = (ApxPrivate *)a->viv;

	if (p->surface_wrapper) {
		wl_proxy_wrapper_destroy(p->surface_wrapper);
		p->surface = NULL;
		p->surface_wrapper = NULL;
		p->surface_id = 0;
	}

	if (b->callback.notify) {
        b->callback.notify(b->callback.data);
    }

	free(b);
	return r;
}

static gceSURF_FORMAT apx_format_to_gc(apxPixelFormat format)
{
	switch (format) {
	case APX_PIXELFORMAT_RGB_565:	return gcvSURF_R5G6B5;
	case APX_PIXELFORMAT_RGBA_8888:	return gcvSURF_A8R8G8B8;
	case APX_PIXELFORMAT_RGBX_8888:	return gcvSURF_X8R8G8B8;
	case APX_PIXELFORMAT_RGBA_4444:	return gcvSURF_A4R4G4B4;
	case APX_PIXELFORMAT_RGBX_4444:	return gcvSURF_X4R4G4B4;
	case APX_PIXELFORMAT_UYVY:	return gcvSURF_UYVY;
	case APX_PIXELFORMAT_YUY2:	return gcvSURF_YUY2;
	case APX_PIXELFORMAT_YV12:	return gcvSURF_YV12;
	case APX_PIXELFORMAT_NV12:	return gcvSURF_NV12;
	case APX_PIXELFORMAT_NV16:	return gcvSURF_NV16;

	default:
		error("Unknown format %i", format);
		return gcvSURF_UNKNOWN;
	}
}

static void release_handler(void *data, struct wl_buffer *wl_buffer)
{
	struct apx_buffer *buffer = (struct apx_buffer *)data;
	debug("wayland called buffer release handler");
	debug("in: %p 0: %p 1: %p", wl_buffer, buffer->wl_buf[0], buffer->wl_buf[1]);
	int index = 0;

	if (buffer->wl_buf[0] == wl_buffer)
		index = 0;
	else if (buffer->wl_buf[1] == wl_buffer)
		index = 1;

	if (buffer->locked[index] == 1)
	{
		debug("unlocking buffer %d", index);
	}
	else
	{
		debug("uhm... Buffer %d is already unlocked...", index);
	}

    if (buffer->callback.func)
	{
		debug("Notifying APX buffer unlock listeners");
		buffer->callback.func(buffer, buffer->callback.data);
	}
}

static const struct wl_buffer_listener buffer_listener = {
	.release = release_handler,
};


/**
 * Create Wayland buffer for an apx buffer.
 *
 * @retval	>=0	if success.
 * @retval	<0	if error.
 */
static int create_wl_buffer(struct apx *a, struct apx_buffer *b, gceSURF_FORMAT gc_format)
{
	gcePOOL pool;
	gctUINT32 node;
	gctUINT bytes, width, height;
	gctINT stride;
    ApxPrivate *priv = (ApxPrivate*)a->viv; /* safe. calling code checks for validity of a */

	if (gcoSURF_QueryVidMemNode(b->surf[b->current], &node, &pool, &bytes)
			!= gcvSTATUS_OK) {
		error("gcoSURF_QueryVidMemNode error");
		goto err;
	}

	if (pool == gcvPOOL_USER) {
		gctUINT32 Address[3];

		/* We Lock()/Unlock() to retrieve the physical pointer. This has the side
		 * benefit of updating the physical pointers for the U/V planes, which
		 * some functions such as FilterBlit() use. */
		if (gcoSURF_Lock(b->surf[b->current], Address, NULL/*Memory*/) != gcvSTATUS_OK) {
			error("gcoSURF_Lock error");
			goto err;
		}

		/* Abuse the node pointer :) */
		node = (gctUINT32)Address[0];
		/* TODO: check contiguous? */

		if (gcoSURF_Unlock(b->surf[b->current], NULL/*Memory*/) != gcvSTATUS_OK) {
			error("gcoSURF_Unlock error");
			goto err;
		}
	} else {
		/* Resolve node handle to name */
		gcoHAL_NameVideoMemory(node, &node);
	}

	if (gcoSURF_GetSize(b->surf[b->current], &width, &height, NULL/*Depth*/)
			!= gcvSTATUS_OK) {
		error("gcoSURF_GetSize error");
		goto err;
	}

	if (gcoSURF_GetAlignedSize(b->surf[b->current], NULL/*Width*/, NULL/*Height*/, &stride)
			!= gcvSTATUS_OK) {
		error("gcoSURF_GetAlignedSize error");
		goto err;
	}

	*(volatile int *)&b->locked = 0;	/* Should be already zero, but better safe... */

    b->wl_buf[b->current] = wl_viv_create_buffer(priv->viv, width, height, stride, gc_format,
					gcvSURF_BITMAP, (int32_t)node, pool, bytes);

	if (!b->wl_buf[b->current]) {
		error("wl_viv_create_buffer error");
		goto err;
	}

	if (wl_buffer_add_listener(b->wl_buf[b->current], &buffer_listener, b) < 0) {
		error("wl_buffer_add_listener error");
		goto err;
	}

	return 0;

err:
	return -1;
}

/**
 * Create buffer.
 *
 * This function allocates the memory, too.
 *
 * @param[in]	a	API context.
 * @param[in]	width	In pixels.
 * @param[in]	height	In pixels.
 * @param[in]	format one of
 *  - APX_PIXELFORMAT_RGB_565
 *    - 16 bit with memory layout BGR
 *  - APX_PIXELFORMAT_RGBA_8888
 *    - 32 bit with memory layout BGRA
 *  - APX_PIXELFORMAT_RGBX_8888
 *    - 32 bit with memory layout BGRX
 *  - APX_PIXELFORMAT_RGBA_4444
 *    - 16 bit with memory layout BGRA
 *  - APX_PIXELFORMAT_RGBX_4444
 *    - 16 bit with memory layout BGRX
 *  - APX_PIXELFORMAT_UYVY
 *  - APX_PIXELFORMAT_YUY2
 *  - APX_PIXELFORMAT_YV12
 *  - APX_PIXELFORMAT_NV12
 *  - APX_PIXELFORMAT_NV16
 *
 * @retval	Pointer	to created buffer if success.
 * @retval	NULL	if error.
 * @ingroup buffer_handling
 */
struct apx_buffer *apx_buffer_create_unbuffered(struct apx *a, unsigned width,
					unsigned height, apxPixelFormat format)
{
	const gcePOOL pool = gcvPOOL_DEFAULT;
    const gceSURF_TYPE type = gcvSURF_BITMAP;
	const gctUINT depth = 1;
	struct apx_buffer *b = NULL;
	gceSURF_FORMAT gc_format;

	/* Verify inputs.*/
	if (!a) {
		error("Bad apx pointer");
		return NULL;
	}

	if (width == 0 || height == 0) {
		error("Bad dimensions %ux%u", width, height);
		return NULL;
	}

	gc_format = apx_format_to_gc(format);

	if (gc_format == gcvSURF_UNKNOWN) {
		error("Bad format %u", format);
		return NULL;
	}

	b = calloc(sizeof(struct apx_buffer), 1);

	if (!b) {
		error("calloc error");
		return NULL;
	}

    b->a = a;

    if (gcoSURF_Construct(gcvNULL, width, height, depth, type, gc_format, pool,
				&b->surf[b->current])
			!= gcvSTATUS_OK) {
		error("gcoSURF_Construct error");
		goto err;
	}

    /* Everything ok.*/
	return b;

err:
	apx_buffer_destroy(b, NULL);
	return NULL;
}

/**
 * Create buffer.
 *
 * This function allocates the memory, too. The buffer will be double-buffered
 * and copy the contents of the current buffer to the back-buffer on apx_buffer_commit().
 * If this behavior is undesired use apx_buffer_set_no_copy() to disable content
 * copying while keeping double-buffering.
 *
 * If double buffering is not needed then apx_buffer_create_unbuffered() should
 * be used to create the buffer.
 *
 * @param[in]	a	API context.
 * @param[in]	width	In pixels.
 * @param[in]	height	In pixels.
 * @param[in]	format one of
 *  - APX_PIXELFORMAT_RGB_565
 *    - 16 bit with memory layout BGR
 *  - APX_PIXELFORMAT_RGBA_8888
 *    - 32 bit with memory layout BGRA
 *  - APX_PIXELFORMAT_RGBX_8888
 *    - 32 bit with memory layout BGRX
 *  - APX_PIXELFORMAT_RGBA_4444
 *    - 16 bit with memory layout BGRA
 *  - APX_PIXELFORMAT_RGBX_4444
 *    - 16 bit with memory layout BGRX
 *  - APX_PIXELFORMAT_UYVY
 *  - APX_PIXELFORMAT_YUY2
 *  - APX_PIXELFORMAT_YV12
 *  - APX_PIXELFORMAT_NV12
 *  - APX_PIXELFORMAT_NV16
 *
 * @retval	Pointer	to created buffer if success.
 * @retval	NULL	if error.
 * @ingroup buffer_handling
 */
struct apx_buffer *apx_buffer_create(struct apx *a, unsigned width,
					unsigned height, apxPixelFormat format)
{
	struct apx_buffer *b = apx_buffer_create_unbuffered(a, width, height, format);
	apx_buffer_set_double_buffered(b);

	return b;
}

/**
 * Wrap buffer.
 *
 * This function "wraps" a surface into an apx buffer.
 *
 * @param[in]	a	API context.
 *
 * @param[in]	p	Pointer to one to three plane pointers, physical
 *			address(es). For RGB formats, only the first pointer is
 *			relevant. For NV12, p[0] is Y and p[1] is UV. For YV12
 *			p[0] is Y, p[1] is V and p[2] is U.
 *
 * @param[in]	l	Pointer to one to three plane pointers, logical
 *			address(es). For RGB formats, only the first pointer is
 *			relevant. For NV12, l[0] is Y and l[1] is UV. For YV12
 *			l[0] is Y, l[1] is V and l[2] is U.
 *
 * @param[in]	width	In pixels.
 * @param[in]	height	In pixels.
 *
 * @param[in]	stride	This is the number of bytes in one horizontal line. See
 *		`apx_buffer_get_stride()`.
 *
 * @param[in]	format one of
 *  - APX_PIXELFORMAT_RGB_565
 *    - 16 bit with memory layout BGR
 *  - APX_PIXELFORMAT_RGBA_8888
 *    - 32 bit with memory layout BGRA
 *  - APX_PIXELFORMAT_RGBX_8888
 *    - 32 bit with memory layout BGRX
 *  - APX_PIXELFORMAT_RGBA_4444
 *    - 16 bit with memory layout BGRA
 *  - APX_PIXELFORMAT_RGBX_4444
 *    - 16 bit with memory layout BGRX
 *  - APX_PIXELFORMAT_UYVY
 *  - APX_PIXELFORMAT_YUY2
 *  - APX_PIXELFORMAT_YV12
 *  - APX_PIXELFORMAT_NV12
 *  - APX_PIXELFORMAT_NV16
 *
 * @retval	Pointer	to created buffer if success.
 * @retval	NULL	if error.
 * @ingroup buffer_handling
 */
struct apx_buffer *apx_buffer_wrap(struct apx *a, void *const *p, void *const *l,
					unsigned width, unsigned height,
					unsigned stride, apxPixelFormat format)
{
	const gceSURF_TYPE type = gcvSURF_BITMAP;
	gceSURF_FORMAT gc_format;
	struct apx_buffer *b = NULL;
	gctUINT32 Planes[3] = {0, 0, 0};	/* used only for alignment check */
	gctUINT32 Strides[3] = {0, 0, 0};
    int status = gcvSTATUS_OK;
    gco2D engine = gcvNULL;

	/* Verify inputs. */
	if (!a) {
		error("Bad apx pointer");
		goto err;
	}

	if (!p && !l) {
		error("Bad pointers");
		goto err;
	}

	if (width == 0 || height == 0) {
		error("Bad dimensions %ux%u", width, height);
		goto err;
	}

	if (stride < width) {
		error("Bad stride %u", stride);
		goto err;
	}

	gc_format = apx_format_to_gc(format);

	if (gc_format == gcvSURF_UNKNOWN) {
		error("Bad format %u", format);
		goto err;
	}

	/* TODO! Check constraints */
	switch (gc_format) {
	/* 1 plane. */
	case gcvSURF_R5G6B5:
	case gcvSURF_A8R8G8B8:
	case gcvSURF_X8R8G8B8:
	case gcvSURF_A4R4G4B4:
	case gcvSURF_X4R4G4B4:
	case gcvSURF_UYVY:
	case gcvSURF_YUY2:
		Strides[0] = stride;

		if (p)
			Planes[0] = (gctUINT32) (p[0]);

		if (l)
			Planes[0] |= (gctUINT32) (l[0]);

		break;

	/* 2 planes. */
	case gcvSURF_NV12:
	case gcvSURF_NV16:
		Strides[0] = Strides[1] = stride;

		if (p) {
			Planes[0] = (gctUINT32) (p[0]);
			Planes[1] = (gctUINT32) (p[1]);
		}

		if (l) {
			Planes[0] |= (gctUINT32) (l[0]);
			Planes[1] |= (gctUINT32) (l[1]);
		}

		break;

	/* 3 planes. */
	case gcvSURF_YV12:
		Strides[0] = stride;
		Strides[1] = Strides[2] = stride / 2;

		if (p) {
			Planes[0] = (gctUINT32) (p[0]);
			Planes[1] = (gctUINT32) (p[1]);
			Planes[2] = (gctUINT32) (p[2]);
		}

		if (l) {
			Planes[0] |= (gctUINT32) (l[0]);
			Planes[1] |= (gctUINT32) (l[1]);
			Planes[2] |= (gctUINT32) (l[2]);
		}

		break;

	default:
		error("Unknown format %i", gc_format);
		goto err;
	}


    status = gcoHAL_Get2DEngine(gcvNULL, &engine);
    if (status != gcvSTATUS_OK)
    {
        error("gcoHAL_Get2DEngine: Failed to get engine: %d", status);
        goto err;
    }

    /* Is the pixmap properly aligned? */
    status = gco2D_CheckSurface(engine, gcvTRUE, gc_format, Planes, Strides,
                                width, height, gcvSURF_0_DEGREE, gcvLINEAR);
    if (status != gcvSTATUS_OK) {
        error("gco2D_CheckSurface error: %d", status);
		goto err;
	}

	b = calloc(sizeof(struct apx_buffer), 1);

	if (!b) {
		error("calloc error");
		goto err;
	}

	b->a = a;

    status = gcoSURF_ConstructWrapper(gcvNULL, &b->surf[b->current]);
    if (status != gcvSTATUS_OK) {
        error("gcoSURF_ConstructWrapper error: %d", status);
		goto err;
	}

	/* Note: SetBuffer() takes a single physical pointer. This imposes the
	 * limitation on contiguous buffers. */
    status = gcoSURF_SetBuffer(b->surf[b->current], type, gc_format, stride, (l ? l[0] : NULL),
            (p ? (gctUINT32)(p[0]) : 0));
    if (status != gcvSTATUS_OK) {
        error("gcoSURF_SetBuffer error: %d", status);
		goto err;
	}

    status = gcoSURF_SetWindow(b->surf[b->current], 0/*X*/, 0/*Y*/, width, height);
    if (status != gcvSTATUS_OK) {
        error("gcoSURF_SetWindow error: %d", status);
		goto err;
	}

	/* TODO check physical pointers from user against computed
	 * Lock, check logical and physical, Unlock */

	/* Everything ok. */
	return b;

err:
	if (b)
		apx_buffer_destroy(b, NULL);

	return NULL;
}

/**
 * Get the stride of the buffer.
 *
 * The stride is the number of bytes for one image line. Due to padding
 * constraints, this may be a bit more than the minimum number of bytes
 * necessary to hold one line.
 *
 * @param[in]	b	Buffer.
 *
 * @param[out]	stride	Pointer to one to three unsigned to hold the resulting
 *			stride(s). For RGB formats only the first stride is relevant.
 *
 * @retval	>=0	if success.
 * @retval	<0	if error.
 * @ingroup buffer_handling
 */
int apx_buffer_get_stride(struct apx_buffer *b, unsigned *stride)
{
	gctINT Stride;

	if (!b) {
		error("Bad pointer");
		return -1;
	}

	if (gcoSURF_GetAlignedSize(b->surf[b->current], NULL/*Width*/, NULL/*Height*/,
			&Stride) != gcvSTATUS_OK) {
		error("gcoSURF_GetAlignedSize error");
		return -1;
	}

	*stride = Stride;

	/* TODO: For YUV, compute 1 and 2? */

	return 0;
}

/**
 * Get the size of the buffer.
 *
 * @param[in]	b	Buffer.
 * @param[out]	width	Pointer to width. In pixels. Ignored if NULL.
 * @param[out]	height	Pointer to height. In pixels. Ignored if NULL.
 *
 * @retval	>=0	if success.
 * @retval	<0	if error.
 * @ingroup buffer_handling
 */
int apx_buffer_get_size(struct apx_buffer *b, unsigned *width,
			unsigned *height)
{
	if (!b) {
		error("Bad pointer");
		return -1;
	}

	if (gcoSURF_GetSize(b->surf[b->current], width, height, NULL/*Depth*/)
		!= gcvSTATUS_OK) {
		error("gcoSURF_GetSize error");
		return -1;
	}

	return 0;
}

static apxPixelFormat gc_format_to_apx(gceSURF_FORMAT format)
{
	switch (format) {
	case gcvSURF_R5G6B5:	return APX_PIXELFORMAT_RGB_565;
	case gcvSURF_A8R8G8B8:	return APX_PIXELFORMAT_RGBA_8888;
	case gcvSURF_X8R8G8B8:	return APX_PIXELFORMAT_RGBX_8888;
	case gcvSURF_A4R4G4B4:	return APX_PIXELFORMAT_RGBA_4444;
	case gcvSURF_X4R4G4B4:	return APX_PIXELFORMAT_RGBX_4444;
	case gcvSURF_UYVY:	return APX_PIXELFORMAT_UYVY;
	case gcvSURF_YUY2:	return APX_PIXELFORMAT_YUY2;
	case gcvSURF_YV12:	return APX_PIXELFORMAT_YV12;
	case gcvSURF_NV12:	return APX_PIXELFORMAT_NV12;
	case gcvSURF_NV16:	return APX_PIXELFORMAT_NV16;

	default:
		error("Unknown format %i", format);
		return APX_PIXEL_FORMAT_UNKNOWN;
	}
}

/**
 * Get the format of the buffer.
 *
 * @param[in]	b	Buffer.
 *
 * @retval	Format				if success.
 * @retval	APX_PIXEL_FORMAT_UNKNOWN	if error.
 * @ingroup buffer_handling
 */
apxPixelFormat apx_buffer_get_format(struct apx_buffer *b)
{
	gceSURF_FORMAT Format;

	if (!b) {
		error("Bad pointer");
		return APX_PIXEL_FORMAT_UNKNOWN;
	}

	if (gcoSURF_GetFormat(b->surf[b->current], NULL, &Format) != gcvSTATUS_OK) {
		error("gcoSURF_GetFormat error");
		return APX_PIXEL_FORMAT_UNKNOWN;
	}

	return gc_format_to_apx(Format);
}

/**
 * Get the corresponding Wayland buffer.
 *
 * @param[in]	b	Buffer.
 *
 * @retval	Pointer	to Wayland buffer if success.
 * @retval	NULL	if error.
 * @ingroup buffer_handling
 */
struct wl_buffer *apx_buffer_get_wl(struct apx_buffer *b)
{
	if (!b) {
		error("Bad pointer");
		return NULL;
	}

	return b->wl_buf[b->current];
}

static void
frame_callback(void *data, struct wl_callback *callback, uint32_t time)
{
	int *done = data;
	(void)time;

	*done = 1;
	wl_callback_destroy(callback);
}

static const struct wl_callback_listener frame_callback_listener = {
	frame_callback
};

static int
set_frame_callback(struct apx *a)
{
	ApxPrivate *p = (ApxPrivate*)a->viv;

	p->redraw_done = 0;
	p->frame_callback = wl_surface_frame(p->surface_wrapper);
	if (p->frame_callback == NULL) {
		return -1;
	}

	wl_callback_add_listener(p->frame_callback, &frame_callback_listener, &p->redraw_done);

	return 0;
}

static void
set_opaque_region(struct wl_surface *wl_surface, int width, int height, struct wl_compositor* compositor)
{
	struct wl_region *region;

	region = wl_compositor_create_region(compositor);
	wl_region_add(region, 0, 0, width, height);
	wl_surface_set_opaque_region(wl_surface, region);
	wl_region_destroy(region);
}
/**
 * Commit buffer.
 *
 * Attach an apx buffer to a Wayland surface (on top-left corner) and commit
 * it.
 *
 * If the buffer was previously mapped, it is unmapped.
 *
 * @param[in]   a   Global apx handle.
 * @param[in]	b	Buffer.
 * @param[in]	s	Pointer to Wayland surface.
 * @param[in]	x	Top-left corner of damaged zone, x coordinate.
 * @param[in]	y	Top-left corner of damaged zone, y coordinate.
 * @param[in]	width	Damaged zone width.
 * @param[in]	height	Damaged zone height.
 *
 * @param[in]	d	Pointer to Wayland display. Ignored if NULL. If
 *			non-NULL, we wait until we receive the frame `done`
 *			event.
 *
 * @retval	>=0	if success.
 * @retval	<0	if error.
 * @ingroup buffer_handling
 */
int apx_buffer_commit(struct apx *a, struct apx_buffer *b, struct wl_surface *s, int x, int y,
			int width, int height, struct wl_display *d)
{
	struct wl_buffer *wl_buf;
	ApxPrivate *priv;

	debug("buffer: %p, surface: %p", b, s);

    if (!a) {
		error("Missing APX pointer");
        goto err;
    }

	if (!d) {
		error("Bad wl_display");
		goto err;
	}

	if (!b || !s) {
		error("Bad pointer(s)");
		goto err;
    }

	if (x < 0 || y < 0) {
		error("Bad top-left corner coordinate(s)");
		goto err;
	}

	if (width < 0 || height < 0) {
		error("Bad surface dimension(s)");
		goto err;
	}

	/* Unmap buffer if necessary. */
	if (b->memory[0] && apx_buffer_unmap(b) < 0) {
		error("apx_buffer_unmap error");
		goto err;
	}

	priv = (ApxPrivate*)a->viv;

	/* Set wl_context if necessary. */
	if (!priv->display && set_wl_context(a, d) < 0) {
		error("set_wl_context error");
		goto err;
	}

	wl_buf = apx_buffer_get_wl(b);
	if (!wl_buf) {
        gceSURF_FORMAT gc_format = gcvSURF_UNKNOWN;

		gcoSURF_GetFormat(b->surf[b->current], gcvNULL, &gc_format);
        if (create_wl_buffer(a, b, gc_format) < 0) {
			error("No wl_buf");
            goto err;
        }
        wl_buf = apx_buffer_get_wl(b);
	}

	/* Mark buffer as locked */
	b->locked[b->current] = 1;

	/* Synchronize the wayland connection */
	if (wayland_sync(a) < 0) {
		error("wayland sync error");
		goto err;
	}

	/* There is no usecase to change the wl_surface dynamically,
	 * so wl_surface_proxy_wrapper is created when commit is
	 * done for the first time */
	if (!priv->surface && set_wl_surface(a, s, width, height) < 0) {
		error("set_wl_surface error");
		goto err;
	}
	else if ((priv->surface != s)||
			(priv->surface_id != wl_proxy_get_id((struct wl_proxy*)s))) {
		error("changing of wl_surf to the apx_context:0x%x not possible wo apx_buffer_destroy",
				(unsigned int)a);
		goto err;
	}

	if (set_frame_callback(a) < 0) {
		error("set_frame_callback error");
		goto err;
	}

	/* Post */
	wl_surface_attach(s, wl_buf, 0, 0);
	wl_surface_damage(s, x, y, width, height);
	wl_surface_commit(s);

	if (b->double_buffered)
	{
		debug("Buffer is double-buffered, copying contents from %p to %p",
			  b->surf[b->current], b->surf[(b->current + 1) % 2]);
        swap_and_blit((ApxPrivate *)a->viv, b);
	}

	if (wl_display_flush(priv->display) < 0) {
		error("wl_display_flush error");
		goto err;
	}

	return 0;

err:
	return -1;
}

/**
 * Install call-back function called when compositor releases buffer.
 *
 * After apx_buffer_commit() the buffer should be considered read-only and not
 * modified to prevent on-screen rendering artifacts. The compositor will notify
 * the client when the buffer is not on-screen anymore. This function can be
 * used to register a callback to get a notification when the buffer is "free"
 * again.
 *
 * @param[in] b struct apx_buffer to get the notification for
 * @param[in] callback function to call on unlock event.
 * @param[in] data user data passed to function. May be NULL.
 * @param[in] notify function that is called on data if b is destroyed. May be NULL.
 * @ingroup buffer_handling
 */
void apx_buffer_set_unlock_callback(struct apx_buffer *b, ApxUnlockCallback callback, void *data, ApxCallbackDestroyNotify notify)
{
    if (b != NULL)
    {
        b->callback.data = data;
        b->callback.func = callback;
        b->callback.notify = notify;
    }
    else
    {
        error("Bad buffer!");
    }
}

/**
 * Obtain the user data set in apx_buffer_commit() call
 *
 *
 * @param[in] b struct apx_buffer to get the user data for
 * @retval = NULL if buffer is NULL
 * @retval = user set value if buffer is non-null
 * @ingroup buffer_handling
 */

void * apx_buffer_get_user_data(struct apx_buffer *b)
{
    void *ret = NULL;
    if (b != NULL)
        ret = b->callback.data;
    return ret;
}
/**
 * Make APX buffer double-buffered.
 *
 * Will create a backbuffer with identical parameters to the original buffer.
 * on apx_buffer_commit() the two buffers will be swapped and contents from the
 * current buffer will be copied to the backbuffer to support implementations
 * that only update deltas not full frames.
 *
 * @note To disable copying while keeping backbuffer functionality, use
 * apx_buffer_set_no_copy().
 *
 * @param[in] b struct apx_buffer to create an internal backbuffer for.
 * @retval >=0 if success.
 * @retval <0 if error.
 * @ingroup buffer_handling
 */
int apx_buffer_set_double_buffered(struct apx_buffer *b)
{
	gcePOOL pool;
	gctUINT width, height, depth;
	gceSURF_FORMAT gc_format;
	gctUINT32 node;
	gctSIZE_T bytes;

    if (b == NULL) {
        error("Bad buffer!");
        return -1;
    }

    gcoSURF_GetFormat(b->surf[b->current], gcvNULL, &gc_format);
    if (gc_format == gcvSURF_UYVY || gc_format == gcvSURF_YUY2 ||
        gc_format == gcvSURF_YV12 || gc_format == gcvSURF_NV12 ||
        gc_format == gcvSURF_NV16)
    {
        error("Double-buffering not supported for YUV buffers.");
        return 0;
    }

	if (gcoSURF_QueryVidMemNode(b->surf[b->current], &node, &pool, &bytes)
			!= gcvSTATUS_OK) {
		error("gcoSURF_QueryVidMemNode error");
		return -1;
	}

	if (pool == gcvPOOL_USER) {
		error("Cannot set wrapped buffers double buffered");
		return -1;
	}

	if (gcoSURF_GetSize(b->surf[b->current], &width, &height, &depth) != gcvSTATUS_OK) {
		error("gcoSURF_GetSize error");
		goto err;
	}

	gcoSURF_GetFormat(b->surf[b->current], gcvNULL, &gc_format);

	b->double_buffered = gcvTRUE;
	/* Create backbuffer */
    if (gcoSURF_Construct(NULL, width, height, depth, gcvSURF_BITMAP | gcvSURF_NO_TILE_STATUS, gc_format,
						  gcvPOOL_DEFAULT, &b->surf[1]) != gcvSTATUS_OK)
	{
        error("Failed to create backbuffer");
		return -1;
	}

	return 0;
err:
	gcoSURF_Destroy(b->surf[1]);

	return -1;
}

/**
 * Set double-buffered buffer to not copy contents on commit.
 *
 * To support use-cases which only render deltas to the previous a
 * double-buffered struct apx_buffer will always copy the contents from the
 * buffer that was send to the compositor to the backbuffer.
 *
 * This function disables the copy while retaining the double-buffering.
 *
 * @param b struct apx_buffer to modify.
 * @ingroup buffer_handling
 */
void apx_buffer_set_no_copy(struct apx_buffer *b)
{
    if (!b)
    {
        error("Bad pointer!");
        return;
    }

    /* only ever necessary if buffer is double-buffered */
    if (b->double_buffered)
    {
        b->no_copy = gcvTRUE;
    }
}
